# Sistema de Design: Website Dinâmico Tjuridico.com.br

## 1. Abordagem de Implementação

Vamos implementar as seguintes tarefas:

1. **Frontend Arquitetura** - React 18 com TypeScript, Tailwind CSS, Shadcn-ui para componentes base
2. **Sistema de Animações** - Three.js/React Three Fiber para 3D, Framer Motion para microinterações, Particle.js para efeitos
3. **Agentes de IA** - Arquitetura multi-agente com FastAPI backend, integração OpenAI/Claude com fine-tuning jurídico
4. **Backend Services** - FastAPI com SQLModel para ORM, Redis para cache, Celery para tarefas assíncronas
5. **Banco de Dados** - PostgreSQL com schema otimizado para gestão de leads e histórico de conversas
6. **Integrações** - WhatsApp Business API, Google Analytics, sistemas internos Tjuridico
7. **Infraestrutura** - Docker, Nginx, CDN para assets, monitoramento com Prometheus/Grafana

## 2. Padrões Principais de Interação Usuário-UI

1. **Navegação com Efeitos de Hover** - Menu com glassmorphism e transições suaves
2. **Chatbot Flutuante** - Ícone animado que se expande em interface completa
3. **Cards 3D Interativos** - Elementos que giram ao hover para mostrar detalhes
4. **Comparador de Planos** - Sliders e toggles com feedback visual imediato
5. **Formulários Dinâmicos** - Campos que se expandem e validam em tempo real
6. **Timeline Narrativa** - Animações acionadas por scroll que contam a história
7. **Carrossel de Depoimentos** - Transições futuristas entre cards
8. **Background de Partículas** - Efeitos dinâmicos que respondem ao cursor

## 3. Arquitetura do Sistema

```plantuml
@startuml
!define RECTANGLE class
skinparam class {
  BackgroundColor White
  BorderColor Black
  ArrowColor Black
}

package "Frontend" {
  [React App] as frontend
  [Three.js / R3F] as threejs
  [Framer Motion] as framer
  [Tailwind CSS] as tailwind
  [Shadcn-ui] as shadcn
}

package "Backend API" {
  [FastAPI Server] as fastapi
  [AI Agents Service] as ai_service
  [Lead Management] as lead_service
  [Analytics Service] as analytics
}

package "Banco de Dados" {
  [PostgreSQL] as postgres
  [Redis Cache] as redis
}

package "Serviços Externos" {
  [OpenAI/Claude API] as llm_api
  [WhatsApp Business API] as whatsapp
  [Google Analytics] as ga
  [CDN Assets] as cdn
}

package "Infraestrutura" {
  [Docker Containers] as docker
  [Nginx Load Balancer] as nginx
  [Monitoring] as monitoring
}

frontend --> fastapi : REST API / WebSocket
threejs --> cdn : Load 3D assets
framer --> frontend : Animation hooks
tailwind --> frontend : Styling
shadcn --> frontend : UI Components

fastapi --> postgres : CRUD Operations
fastapi --> redis : Session/Cache
fastapi --> ai_service : Agent Requests
ai_service --> llm_api : LLM Calls
lead_service --> whatsapp : Notifications
analytics --> ga : Track Events

docker --> nginx : Reverse Proxy
nginx --> frontend : Serve Static
nginx --> fastapi : Proxy API
monitoring --> docker : Metrics

@enduml
```

## 4. Fluxo de Navegação UI

```plantuml
@startuml
state "Home" as Home {
  [*] --> HeroSection
  HeroSection --> ServicesSection : Scroll Down
  HeroSection --> Chatbot : Click Chat Icon
}

state "Serviços" as ServicesSection {
  [*] --> ServicesGrid
  ServicesGrid --> ServiceDetail : Click Card
  ServiceDetail --> ServicesGrid : Back
}

state "Planos" as PlansSection {
  [*] --> PlansComparison
  PlansComparison --> PlanDetail : Select Plan
  PlanDetail --> ContactForm : Click "Contratar"
}

state "Contato" as ContactSection {
  [*] --> ContactOptions
  ContactOptions --> Chatbot : "Falar com IA"
  ContactOptions --> ContactForm : "Formulário"
  ContactOptions --> WhatsApp : "WhatsApp"
}

state "Chatbot" as ChatbotState {
  [*] --> WelcomeMessage
  WelcomeMessage --> Qualification : User Responds
  Qualification --> LegalExpert : "Dúvida Técnica"
  Qualification --> LeadCapture : "Interesse"
  LegalExpert --> HumanEscalation : "Falar com Advogado"
  LeadCapture --> ThankYou : Lead Captured
}

Home --> ServicesSection : Menu "Serviços"
Home --> PlansSection : Menu "Planos"
Home --> ContactSection : Menu "Contato"
ServicesSection --> Home : Logo Click
PlansSection --> Home : Logo Click
ContactSection --> Home : Logo Click
ChatbotState --> Home : Close Chat

@enduml
```

## 5. Diagrama de Classes

```plantuml
@startuml
@startuml
interface IAnimationService {
  +playScrollAnimation(elementId: string, triggerPoint: number): void
  +playHoverAnimation(elementId: string): Promise<void>
  +initializeParticles(config: ParticleConfig): void
}

interface IAIAgent {
  +processMessage(message: string, context: ChatContext): Promise<AgentResponse>
  +escalateToHuman(sessionId: string, reason: string): Promise<void>
  +updateKnowledgeBase(content: LegalContent): Promise<void>
}

class AnimationManager {
  +scrollAnimations: Map<string, ScrollAnimation>
  +hoverAnimations: Map<string, HoverAnimation>
  +particleSystems: ParticleSystem[]
  +registerScrollTrigger(element: HTMLElement, callback: Function): void
  +playSequence(animationSequence: AnimationSequence): Promise<void>
}

class ThreeJSAnimation {
  +scene: THREE.Scene
  +camera: THREE.Camera
  +renderer: THREE.WebGLRenderer
  +loadLegalModel(modelType: LegalModelType): Promise<THREE.Object3D>
  +animateOnScroll(object: THREE.Object3D, scrollY: number): void
  +dispose(): void
}

class ChatbotManager {
  +activeSession: ChatSession | null
  +aiAgents: Map<AgentType, IAIAgent>
  +messageHistory: Message[]
  +sendMessage(content: string): Promise<ChatResponse>
  +transferToAgent(agentType: AgentType): Promise<void>
  +endSession(): void
}

class QualificationAgent implements IAIAgent {
  +qualificationQuestions: Question[]
  +leadScore: number
  +processMessage(message: string, context: ChatContext): Promise<AgentResponse>
  +calculateLeadScore(responses: UserResponse[]): number
  +escalateToHuman(sessionId: string, reason: string): Promise<void>
}

class LegalExpertAgent implements IAIAgent {
  +legalKnowledgeBase: LegalDocument[]
  +jurisdiction: Jurisdiction
  +processMessage(message: string, context: ChatContext): Promise<AgentResponse>
  +searchLegalPrecedents(query: string): LegalPrecedent[]
  +escalateToHuman(sessionId: string, reason: string): Promise<void>
}

class LeadService {
  +leads: Lead[]
  +createLead(data: LeadData): Promise<Lead>
  +updateLeadStatus(leadId: string, status: LeadStatus): Promise<void>
  +qualifyLead(leadId: string, score: number): Promise<void>
  +getLeadsByStatus(status: LeadStatus): Promise<Lead[]>
}

class Lead {
  +id: string
  +name: string
  +email: string
  +company: string
  +phone: string
  +leadScore: number
  +status: LeadStatus
  +conversationHistory: Message[]
  +createdAt: Date
  +updatedAt: Date
}

class ChatSession {
  +id: string
  +userId: string | null
  +messages: Message[]
  +currentAgent: AgentType
  +startTime: Date
  +endTime: Date | null
  +addMessage(message: Message): void
  +getMessageHistory(): Message[]
}

IAnimationService <|.. AnimationManager
IAnimationService <|.. ThreeJSAnimation
IAIAgent <|.. QualificationAgent
IAIAgent <|.. LegalExpertAgent

ChatbotManager --> IAIAgent : uses
ChatbotManager --> ChatSession : manages
LeadService --> Lead : manages
QualificationAgent --> LeadService : updates

@enduml
```

## 6. Diagrama de Sequência

```plantuml
@startuml
actor Usuário
participant "Frontend React" as UI
participant "Animation Service" as Anim
participant "Chatbot API" as ChatAPI
participant "Qualification Agent" as QualAgent
participant "Lead Service" as LeadSvc
participant "Database" as DB

Usuário -> UI: Acessa website
UI -> Anim: Inicializa animações
note right
  Configurações:
  - Partículas background
  - Scroll listeners
  - Hover effects
end note
Anim --> UI: Animações prontas

Usuário -> UI: Clica no chatbot
UI -> ChatAPI: POST /chat/session
ChatAPI -> DB: Cria nova sessão
DB --> ChatAPI: Session ID
ChatAPI --> UI: Sessão criada

UI -> ChatAPI: POST /chat/message
note right
  Input: {
    "sessionId": "abc123",
    "message": "Preciso de ajuda com contrato",
    "context": {
      "page": "services",
      "timeOnSite": 120
    }
  }
end note
ChatAPI -> QualAgent: processMessage()
QualAgent -> QualAgent: Analisa intenção
QualAgent -> QualAgent: Busca perguntas qualificação

QualAgent -> ChatAPI: Resposta com pergunta
note right
  Output: {
    "response": "Entendi que precisa de ajuda com contrato...",
    "nextStep": "qualification",
    "questions": ["Tipo de empresa?", "Número de funcionários?"]
  }
end note
ChatAPI --> UI: Mensagem do agente

Usuário -> UI: Responde perguntas
UI -> ChatAPI: POST /chat/message (respostas)
ChatAPI -> QualAgent: processMessage(respostas)
QualAgent -> QualAgent: Calcula lead score
QualAgent -> LeadSvc: createLead()
LeadSvc -> DB: Insere lead qualificado
DB --> LeadSvc: Lead ID
LeadSvc --> QualAgent: Lead criado

QualAgent -> ChatAPI: Resposta final
note right
  Output: {
    "response": "Ótimo! Seu perfil foi qualificado...",
    "leadScore": 85,
    "recommendedPlan": "Premium",
    "nextActions": ["Agendar consulta", "Ver planos"]
  }
end note
ChatAPI --> UI: Recomendações personalizadas

@enduml
```

## 7. Diagrama ER do Banco de Dados

```plantuml
@startuml
entity "leads" {
  * id : uuid <<PK>>
  --
  * full_name : varchar(255)
  * email : varchar(255)
  company_name : varchar(255)
  phone : varchar(20)
  * lead_score : integer
  * status : lead_status_enum
  source : varchar(100)
  * created_at : timestamp
  * updated_at : timestamp
  conversation_id : uuid <<FK>>
}

entity "conversations" {
  * id : uuid <<PK>>
  --
  * session_id : varchar(255)
  * started_at : timestamp
  ended_at : timestamp
  * initial_page : varchar(100)
  user_agent : text
  ip_address : inet
  * metadata : jsonb
}

entity "messages" {
  * id : uuid <<PK>>
  --
  * conversation_id : uuid <<FK>>
  * message_type : message_type_enum
  * content : text
  * sender : sender_enum
  * timestamp : timestamp
  metadata : jsonb
}

entity "ai_agents" {
  * id : uuid <<PK>>
  --
  * agent_type : agent_type_enum
  * name : varchar(100)
  * description : text
  * is_active : boolean
  * config : jsonb
  * created_at : timestamp
  updated_at : timestamp
}

entity "legal_knowledge" {
  * id : uuid <<PK>>
  --
  * title : varchar(255)
  * content : text
  * category : knowledge_category_enum
  * jurisdiction : varchar(100)
  * tags : text[]
  * created_at : timestamp
  * updated_at : timestamp
  version : integer
}

entity "animation_logs" {
  * id : uuid <<PK>>
  --
  * user_id : varchar(255)
  * animation_type : varchar(100)
  * element_id : varchar(255)
  * trigger_type : trigger_enum
  * timestamp : timestamp
  duration_ms : integer
  success : boolean
}

leads }o--|| conversations : "tem conversa"
messages }o--|| conversations : "pertence a"
ai_agents ||..o{ messages : "processa"

note right of leads
  Status possíveis:
  - new
  - contacted
  - qualified
  - converted
  - rejected
end note

note right of messages
  Tipos de mensagem:
  - user_message
  - ai_response
  - system_notification
  - human_transfer
end note

@enduml
```

## 8. Aspectos NÃO CLAROS / Suposições

1. **Integração com Sistemas Existentes** - Suponho que a Tjuridico possui CRM interno que precisará de API para integração
2. **Modelos de IA Específicos** - Assumo uso de GPT-4 ou Claude com fine-tuning para português jurídico brasileiro
3. **Requisitos de Compliance** - Suponho necessidade de logging completo e consentimento para coleta de dados
4. **Orçamento para APIs de IA** - Assumo orçamento para uso comercial de APIs de LLM (OpenAI/Anthropic)
5. **Equipe de Manutenção** - Suponho que haverá equipe dedicada para treinamento contínuo dos agentes
6. **Escalabilidade de Animações** - Assumo que dispositivos móveis terão versões simplificadas das animações
7. **Backup e Recovery** - Suponho necessidade de sistema de backup para dados de leads e conversas

## 9. Decisões Técnicas Principais

1. **Frontend Framework**: React 18 com TypeScript para type safety e melhor manutenção
2. **Animação 3D**: Three.js com React Three Fiber para integração React nativa
3. **Microanimações**: Framer Motion pela simplicidade e performance
4. **Backend**: FastAPI pela performance assíncrona nativa e documentação automática
5. **Banco de Dados**: PostgreSQL com JSONB para flexibilidade em dados de conversa
6. **Cache**: Redis para sessões e cache de respostas frequentes da IA
7. **Orquestração**: Docker Compose para ambiente de desenvolvimento consistente
8. **Monitoramento**: Prometheus + Grafana para métricas de performance e uso