import { useEffect, useRef } from 'react';
import { ArrowRight, Zap, Shield, Brain, TrendingUp, Users, Clock, CheckCircle } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Card, CardContent } from '@/components/ui/card';
import Hero from '@/components/sections/Hero';
import Services from '@/components/sections/Services';
import AIAgents from '@/components/sections/AIAgents';
import Plans from '@/components/sections/Plans';
import Contact from '@/components/sections/Contact';
import About from '@/components/sections/About';
import Testimonials from '@/components/sections/Testimonials';

const Home = () => {
  const statsRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const observer = new IntersectionObserver(
      (entries) => {
        entries.forEach((entry) => {
          if (entry.isIntersecting) {
            entry.target.classList.add('animate-in', 'fade-in');
          }
        });
      },
      { threshold: 0.1 }
    );

    if (statsRef.current) {
      observer.observe(statsRef.current);
    }

    return () => observer.disconnect();
  }, []);

  const stats = [
    { icon: Users, value: '500+', label: 'Clientes Atendidos', color: 'from-tjuridico-cyan to-tjuridico-purple' },
    { icon: CheckCircle, value: '98%', label: 'Satisfação', color: 'from-tjuridico-green to-tjuridico-cyan' },
    { icon: Clock, value: '24/7', label: 'Disponibilidade', color: 'from-tjuridico-purple to-tjuridico-green' },
    { icon: TrendingUp, value: '60%', label: 'Eficiência', color: 'from-tjuridico-cyan to-tjuridico-green' },
  ];

  const features = [
    {
      icon: Brain,
      title: 'Inteligência Artificial',
      description: 'Agentes de IA especializados em direito brasileiro para análise e consultoria.',
      color: 'text-tjuridico-cyan',
      bgColor: 'bg-tjuridico-cyan/10',
    },
    {
      icon: Shield,
      title: 'Segurança Máxima',
      description: 'Criptografia de ponta a ponta e compliance total com a LGPD.',
      color: 'text-tjuridico-green',
      bgColor: 'bg-tjuridico-green/10',
    },
    {
      icon: Zap,
      title: 'Velocidade',
      description: 'Processamento de documentos em minutos, não em dias.',
      color: 'text-tjuridico-purple',
      bgColor: 'bg-tjuridico-purple/10',
    },
  ];

  return (
    <div className="relative overflow-hidden">
      {/* Animated background elements */}
      <div className="fixed inset-0 -z-10">
        <div className="absolute top-1/4 left-1/4 w-96 h-96 bg-tjuridico-cyan/5 rounded-full blur-3xl"></div>
        <div className="absolute bottom-1/4 right-1/4 w-96 h-96 bg-tjuridico-purple/5 rounded-full blur-3xl"></div>
      </div>

      <Hero />
      
      {/* Stats Section */}
      <section ref={statsRef} className="py-20">
        <div className="container mx-auto px-6">
          <div className="grid grid-cols-2 md:grid-cols-4 gap-8">
            {stats.map((stat, index) => (
              <div
                key={stat.label}
                className="text-center opacity-0"
                style={{ animationDelay: `${index * 100}ms` }}
              >
                <div className={`inline-flex items-center justify-center w-20 h-20 rounded-2xl bg-gradient-to-br ${stat.color} mb-4`}>
                  <stat.icon className="w-10 h-10 text-white" />
                </div>
                <div className="text-4xl font-bold bg-gradient-to-r from-white to-tjuridico-gray bg-clip-text text-transparent">
                  {stat.value}
                </div>
                <div className="text-tjuridico-gray mt-2">{stat.label}</div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Features Section */}
      <section className="py-20">
        <div className="container mx-auto px-6">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold mb-6">
              <span className="gradient-text">Por que escolher </span>
              <span className="text-white">Tjuridico?</span>
            </h2>
            <p className="text-xl text-tjuridico-gray max-w-3xl mx-auto">
              Combinamos tecnologia de ponta com expertise jurídica para oferecer soluções inovadoras
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            {features.map((feature, index) => (
              <Card
                key={feature.title}
                className="glass card-hover border-white/10 overflow-hidden group"
                style={{ animationDelay: `${index * 200}ms` }}
              >
                <CardContent className="p-8">
                  <div className={`w-16 h-16 ${feature.bgColor} rounded-2xl flex items-center justify-center mb-6 group-hover:scale-110 transition-transform duration-300`}>
                    <feature.icon className={`w-8 h-8 ${feature.color}`} />
                  </div>
                  <h3 className="text-2xl font-bold text-white mb-4">{feature.title}</h3>
                  <p className="text-tjuridico-gray">{feature.description}</p>
                  <div className="mt-6 pt-6 border-t border-white/10">
                    <Button
                      variant="ghost"
                      className="text-tjuridico-gray hover:text-white group-hover:text-tjuridico-cyan transition-colors p-0"
                    >
                      Saiba mais
                      <ArrowRight className="w-4 h-4 ml-2 group-hover:translate-x-1 transition-transform" />
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      <Services />
      <AIAgents />
      <Plans />
      <About />
      <Testimonials />
      <Contact />

      {/* CTA Section */}
      <section className="py-20">
        <div className="container mx-auto px-6">
          <div className="glass rounded-3xl p-8 md:p-12 text-center relative overflow-hidden">
            {/* Background gradient */}
            <div className="absolute inset-0 bg-gradient-to-r from-tjuridico-cyan/10 via-tjuridico-purple/10 to-tjuridico-green/10"></div>
            
            <div className="relative z-10">
              <h2 className="text-4xl md:text-5xl font-bold mb-6">
                <span className="text-white">Pronto para transformar </span>
                <span className="gradient-text">seu escritório?</span>
              </h2>
              <p className="text-xl text-tjuridico-gray mb-8 max-w-2xl mx-auto">
                Junte-se a centenas de escritórios que já estão usando a Tjuridico para revolucionar seus processos jurídicos.
              </p>
              <div className="flex flex-col sm:flex-row gap-4 justify-center">
                <Button size="lg" className="btn-futuristic text-lg px-8 py-6">
                  Começar Agora Gratuitamente
                  <ArrowRight className="w-5 h-5 ml-2" />
                </Button>
                <Button
                  size="lg"
                  variant="outline"
                  className="border-tjuridico-cyan/30 text-tjuridico-cyan hover:bg-tjuridico-cyan/10 text-lg px-8 py-6"
                >
                  Agendar Demonstração
                </Button>
              </div>
              <p className="text-tjuridico-gray text-sm mt-6">
                Teste grátis por 14 dias • Sem necessidade de cartão de crédito
              </p>
            </div>
          </div>
        </div>
      </section>
    </div>
  );
};

export default Home;