import { useEffect, useRef } from 'react';
import { ArrowRight, Play, Sparkles } from 'lucide-react';
import { Button } from '@/components/ui/button';
import ParticleBackground from '@/components/animations/ParticleBackground';

const Hero = () => {
  const heroRef = useRef<HTMLDivElement>(null);
  const textRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const observer = new IntersectionObserver(
      (entries) => {
        entries.forEach((entry) => {
          if (entry.isIntersecting) {
            entry.target.classList.add('animate-in', 'fade-in');
          }
        });
      },
      { threshold: 0.1 }
    );

    if (textRef.current) {
      observer.observe(textRef.current);
    }

    return () => observer.disconnect();
  }, []);

  return (
    <section
      ref={heroRef}
      className="relative min-h-screen flex items-center justify-center overflow-hidden pt-16"
    >
      {/* Particle Background */}
      <ParticleBackground />

      {/* Static Background */}
      <div className="absolute inset-0 z-0">
        <div className="absolute inset-0 bg-gradient-to-b from-tjuridico-black via-tjuridico-midnight to-tjuridico-black"></div>
      </div>

      <div className="container mx-auto px-6 relative z-10">
        <div
          ref={textRef}
          className="max-w-6xl mx-auto text-center opacity-0"
        >
          {/* Badge */}
          <div className="inline-flex items-center space-x-2 bg-gradient-to-r from-tjuridico-cyan/10 to-tjuridico-purple/10 border border-tjuridico-cyan/20 rounded-full px-4 py-2 mb-8 animate-in slide-in-from-bottom-4">
            <Sparkles className="w-4 h-4 text-tjuridico-cyan" />
            <span className="text-tjuridico-cyan font-medium">
              Revolução Legal com IA • 60% mais eficiência
            </span>
          </div>

          {/* Main Heading */}
          <h1 className="text-5xl md:text-7xl lg:text-8xl font-bold mb-8 leading-tight">
            <span className="text-white">O Futuro do </span>
            <span className="gradient-text">Direito</span>
            <span className="text-white"> é </span>
            <span className="relative inline-block">
              <span className="gradient-text">Agora</span>
              <div className="absolute -bottom-2 left-0 right-0 h-1 bg-gradient-to-r from-tjuridico-cyan via-tjuridico-purple to-tjuridico-green rounded-full"></div>
            </span>
          </h1>

          {/* Subheading */}
          <p className="text-xl md:text-2xl text-tjuridico-gray mb-12 max-w-3xl mx-auto leading-relaxed">
            Automatize a revisão e análise de contratos com precisão de 98%
            {' • '}
            <span className="text-tjuridico-cyan font-semibold">
              Aumente sua eficiência em até 60%
            </span>
          </p>

          {/* CTA Buttons */}
          <div className="flex flex-col sm:flex-row gap-4 justify-center mb-16">
            <Button size="lg" className="btn-futuristic text-lg px-8 py-6 group">
              Começar Agora Gratuitamente
              <ArrowRight className="w-5 h-5 ml-2 group-hover:translate-x-1 transition-transform" />
            </Button>
            <Button
              size="lg"
              variant="outline"
              className="border-tjuridico-cyan/30 text-tjuridico-cyan hover:bg-tjuridico-cyan/10 text-lg px-8 py-6 group"
            >
              <Play className="w-5 h-5 mr-2" />
              Ver Demonstração
            </Button>
          </div>

          {/* Stats */}
          <div className="grid grid-cols-2 md:grid-cols-4 gap-6 max-w-3xl mx-auto">
            {[
              { value: '500+', label: 'Clientes Satisfeitos' },
              { value: '24/7', label: 'Suporte IA' },
              { value: '98%', label: 'Precisão' },
              { value: 'R$2M+', label: 'Economizado' },
            ].map((stat, index) => (
              <div
                key={stat.label}
                className="glass rounded-2xl p-4 border border-white/10 hover:border-tjuridico-cyan/30 transition-all duration-300 hover:scale-105"
                style={{ animationDelay: `${index * 200 + 500}ms` }}
              >
                <div className="text-2xl md:text-3xl font-bold text-white mb-1">{stat.value}</div>
                <div className="text-sm text-tjuridico-gray">{stat.label}</div>
              </div>
            ))}
          </div>

          {/* Scroll Indicator */}
          <div className="absolute bottom-8 left-1/2 transform -translate-x-1/2 animate-bounce">
            <div className="w-6 h-10 border-2 border-tjuridico-cyan/50 rounded-full flex justify-center">
              <div className="w-1 h-3 bg-tjuridico-cyan rounded-full mt-2 animate-pulse"></div>
            </div>
          </div>
        </div>
      </div>

      {/* Floating Elements */}
      <div className="absolute top-1/4 left-10 w-4 h-4 bg-tjuridico-cyan rounded-full animate-pulse-glow"></div>
      <div className="absolute bottom-1/4 right-10 w-6 h-6 bg-tjuridico-purple rounded-full animate-pulse-glow animation-delay-1000"></div>
      <div className="absolute top-1/3 right-20 w-3 h-3 bg-tjuridico-green rounded-full animate-pulse-glow animation-delay-500"></div>
    </section>
  );
};

export default Hero;