import { useState } from 'react';
import { Mail, Phone, MapPin, Send, Clock, CheckCircle } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent } from '@/components/ui/card';

const Contact = () => {
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    phone: '',
    company: '',
    message: '',
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    // In a real app, this would send data to a backend
    console.log('Form submitted:', formData);
    alert('Mensagem enviada com sucesso! Entraremos em contato em breve.');
    setFormData({ name: '', email: '', phone: '', company: '', message: '' });
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value,
    });
  };

  const contactInfo = [
    {
      icon: Mail,
      title: 'E-mail',
      details: ['contato@tjuridico.com.br', 'suporte@tjuridico.com.br'],
      color: 'text-tjuridico-cyan',
      bgColor: 'bg-tjuridico-cyan/10',
    },
    {
      icon: Phone,
      title: 'Telefone',
      details: ['(11) 99999-9999', '(11) 88888-8888'],
      color: 'text-tjuridico-purple',
      bgColor: 'bg-tjuridico-purple/10',
    },
    {
      icon: MapPin,
      title: 'Endereço',
      details: ['Av. Paulista, 1000', 'Bela Vista, São Paulo - SP', 'CEP: 01310-100'],
      color: 'text-tjuridico-green',
      bgColor: 'bg-tjuridico-green/10',
    },
    {
      icon: Clock,
      title: 'Horário',
      details: ['Segunda a Sexta: 9h às 18h', 'Plantão: 24/7 por chat'],
      color: 'text-tjuridico-pink',
      bgColor: 'bg-tjuridico-pink/10',
    },
  ];

  const faqs = [
    {
      question: 'Quanto tempo leva para implementar?',
      answer: 'A implementação básica leva menos de 24 horas. Configurações personalizadas podem levar 3-5 dias úteis.',
    },
    {
      question: 'Oferecem treinamento?',
      answer: 'Sim, incluímos treinamento gratuito para toda a equipe. Também temos tutoriais em vídeo e documentação detalhada.',
    },
    {
      question: 'Integração com outros sistemas?',
      answer: 'Integramos com os principais sistemas jurídicos, ERPs e ferramentas de produtividade. Nossa API é aberta para desenvolvedores.',
    },
    {
      question: 'Suporte técnico disponível?',
      answer: 'Oferecemos suporte 24/7 por chat, e-mail e telefone. Temos SLA de resposta em até 15 minutos para questões críticas.',
    },
  ];

  return (
    <section id="contact" className="py-20 relative">
      {/* Background Pattern */}
      <div className="absolute inset-0 opacity-5">
        <div className="absolute inset-0" style={{
          backgroundImage: `url("https://mgx-backend-cdn.metadl.com/generate/images/877184/2026-01-01/dafffe65-b182-4c66-b32a-b3b591a02f91.png")`,
          backgroundSize: 'cover',
          backgroundPosition: 'center',
        }}></div>
      </div>

      <div className="container mx-auto px-6 relative z-10">
        {/* Header */}
        <div className="text-center mb-16">
          <h2 className="text-4xl md:text-5xl font-bold mb-6">
            <span className="text-white">Entre em </span>
            <span className="gradient-text">Contato</span>
          </h2>
          <p className="text-xl text-tjuridico-gray max-w-3xl mx-auto">
            Tem alguma dúvida ou precisa de uma solução personalizada? Nossa equipe está pronta para ajudar.
          </p>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-12">
          {/* Contact Info */}
          <div className="lg:col-span-1">
            <div className="space-y-6 mb-8">
              {contactInfo.map((info, index) => (
                <Card
                  key={info.title}
                  className="glass border-white/10 card-hover"
                  style={{ animationDelay: `${index * 100}ms` }}
                >
                  <CardContent className="p-6">
                    <div className="flex items-start space-x-4">
                      <div className={`w-12 h-12 rounded-xl ${info.bgColor} flex items-center justify-center flex-shrink-0`}>
                        <info.icon className={`w-6 h-6 ${info.color}`} />
                      </div>
                      <div>
                        <h3 className="text-lg font-bold text-white mb-2">{info.title}</h3>
                        {info.details.map((detail, idx) => (
                          <p key={idx} className="text-tjuridico-gray">{detail}</p>
                        ))}
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>

            {/* Quick Stats */}
            <Card className="glass border-white/10">
              <CardContent className="p-6">
                <div className="flex items-center space-x-3 mb-4">
                  <CheckCircle className="w-6 h-6 text-tjuridico-green" />
                  <h3 className="text-lg font-bold text-white">Resposta Rápida</h3>
                </div>
                <div className="space-y-3">
                  <div className="flex justify-between">
                    <span className="text-tjuridico-gray">Tempo médio de resposta:</span>
                    <span className="text-white font-semibold">15 minutos</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-tjuridico-gray">Taxa de resolução:</span>
                    <span className="text-white font-semibold">98%</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-tjuridico-gray">Satisfação do cliente:</span>
                    <span className="text-white font-semibold">4.9/5</span>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Contact Form */}
          <div className="lg:col-span-2">
            <Card className="glass border-white/10">
              <CardContent className="p-8">
                <h3 className="text-2xl font-bold text-white mb-6">Envie sua mensagem</h3>
                <form onSubmit={handleSubmit} className="space-y-6">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <label htmlFor="name" className="block text-sm font-medium text-tjuridico-gray mb-2">
                        Nome completo *
                      </label>
                      <Input
                        id="name"
                        name="name"
                        value={formData.name}
                        onChange={handleChange}
                        required
                        className="bg-white/5 border-white/10 text-white placeholder:text-tjuridico-gray focus:border-tjuridico-cyan"
                        placeholder="Seu nome"
                      />
                    </div>
                    <div>
                      <label htmlFor="email" className="block text-sm font-medium text-tjuridico-gray mb-2">
                        E-mail *
                      </label>
                      <Input
                        id="email"
                        name="email"
                        type="email"
                        value={formData.email}
                        onChange={handleChange}
                        required
                        className="bg-white/5 border-white/10 text-white placeholder:text-tjuridico-gray focus:border-tjuridico-cyan"
                        placeholder="seu@email.com"
                      />
                    </div>
                  </div>

                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <label htmlFor="phone" className="block text-sm font-medium text-tjuridico-gray mb-2">
                        Telefone
                      </label>
                      <Input
                        id="phone"
                        name="phone"
                        value={formData.phone}
                        onChange={handleChange}
                        className="bg-white/5 border-white/10 text-white placeholder:text-tjuridico-gray focus:border-tjuridico-cyan"
                        placeholder="(11) 99999-9999"
                      />
                    </div>
                    <div>
                      <label htmlFor="company" className="block text-sm font-medium text-tjuridico-gray mb-2">
                        Empresa/Escritório
                      </label>
                      <Input
                        id="company"
                        name="company"
                        value={formData.company}
                        onChange={handleChange}
                        className="bg-white/5 border-white/10 text-white placeholder:text-tjuridico-gray focus:border-tjuridico-cyan"
                        placeholder="Nome da empresa"
                      />
                    </div>
                  </div>

                  <div>
                    <label htmlFor="message" className="block text-sm font-medium text-tjuridico-gray mb-2">
                      Mensagem *
                    </label>
                    <Textarea
                      id="message"
                      name="message"
                      value={formData.message}
                      onChange={handleChange}
                      required
                      rows={6}
                      className="bg-white/5 border-white/10 text-white placeholder:text-tjuridico-gray focus:border-tjuridico-cyan resize-none"
                      placeholder="Descreva sua necessidade ou dúvida..."
                    />
                  </div>

                  <div className="flex items-center space-x-4">
                    <Button type="submit" className="btn-futuristic">
                      <Send className="w-4 h-4 mr-2" />
                      Enviar Mensagem
                    </Button>
                    <p className="text-sm text-tjuridico-gray">
                      Entraremos em contato em até 2 horas úteis.
                    </p>
                  </div>
                </form>
              </CardContent>
            </Card>

            {/* FAQ Section */}
            <div className="mt-12">
              <h3 className="text-2xl font-bold text-white mb-6">Perguntas Frequentes</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                {faqs.map((faq, index) => (
                  <Card
                    key={faq.question}
                    className="glass border-white/10 card-hover"
                    style={{ animationDelay: `${index * 100}ms` }}
                  >
                    <CardContent className="p-6">
                      <h4 className="text-lg font-bold text-white mb-2">{faq.question}</h4>
                      <p className="text-tjuridico-gray">{faq.answer}</p>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </div>
          </div>
        </div>

        {/* Map/CTA Section */}
        <div className="mt-16 glass rounded-3xl p-8 text-center">
          <h3 className="text-2xl font-bold text-white mb-4">Agende uma demonstração personalizada</h3>
          <p className="text-tjuridico-gray mb-6 max-w-2xl mx-auto">
            Mostraremos como a Tjuridico pode transformar especificamente seus processos jurídicos.
            Demonstração online de 30 minutos, sem compromisso.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button className="btn-futuristic">
              Agendar Demonstração
            </Button>
            <Button
              variant="outline"
              className="border-tjuridico-cyan/30 text-tjuridico-cyan hover:bg-tjuridico-cyan/10"
            >
              Baixar Whitepaper
            </Button>
            <Button
              variant="outline"
              className="border-tjuridico-purple/30 text-tjuridico-purple hover:bg-tjuridico-purple/10"
            >
              Ver Estudo de Caso
            </Button>
          </div>
        </div>
      </div>
    </section>
  );
};

export default Contact;