import { useState } from 'react';
import { Brain, MessageSquare, Zap, Shield, Users, TrendingUp, ChevronRight } from 'lucide-react';
import { Card, CardContent } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';

const AIAgents = () => {
  const [activeAgent, setActiveAgent] = useState('legal');

  const agents = [
    {
      id: 'legal',
      name: 'Legal Assistant',
      description: 'Especialista em direito brasileiro para consultas e análises',
      icon: Brain,
      color: 'from-tjuridico-cyan to-tjuridico-blue',
      features: ['Análise de contratos', 'Consultas jurídicas', 'Pesquisa de jurisprudência'],
      delay: '0ms',
    },
    {
      id: 'customer',
      name: 'Customer Support',
      description: 'Atendimento 24/7 para clientes do escritório',
      icon: MessageSquare,
      color: 'from-tjuridico-purple to-tjuridico-pink',
      features: ['Respostas instantâneas', 'Agendamento de reuniões', 'Follow-up automático'],
      delay: '100ms',
    },
    {
      id: 'compliance',
      name: 'Compliance Officer',
      description: 'Monitoramento de conformidade e riscos legais',
      icon: Shield,
      color: 'from-tjuridico-green to-tjuridico-cyan',
      features: ['Auditoria LGPD', 'Monitoramento de riscos', 'Relatórios de compliance'],
      delay: '200ms',
    },
    {
      id: 'analytics',
      name: 'Analytics Pro',
      description: 'Análise preditiva e insights para tomada de decisão',
      icon: TrendingUp,
      color: 'from-tjuridico-blue to-tjuridico-purple',
      features: ['Análise de casos', 'Previsão de resultados', 'Otimização de estratégias'],
      delay: '300ms',
    },
  ];

  const agentDetails = {
    legal: {
      image: '/images/LegalTechnology.jpg',
      capabilities: [
        'Análise de documentos legais em segundos',
        'Pesquisa em base de dados jurídica com mais de 1 milhão de casos',
        'Geração de pareceres preliminares',
        'Identificação de riscos contratuais',
      ],
      stats: [
        { label: 'Precisão', value: '98%' },
        { label: 'Tempo de resposta', value: '< 5s' },
        { label: 'Casos analisados', value: '10K+' },
      ],
    },
    customer: {
      image: '/images/LegalTechnology.jpg',
      capabilities: [
        'Atendimento multilingue 24/7',
        'Integração com calendário para agendamentos',
        'Análise de sentimento do cliente',
        'Encaminhamento para especialistas humanos',
      ],
      stats: [
        { label: 'Satisfação', value: '95%' },
        { label: 'Resposta média', value: '2s' },
        { label: 'Clientes atendidos', value: '5K+' },
      ],
    },
    compliance: {
      image: '/images/LegalTechnology.jpg',
      capabilities: [
        'Monitoramento contínuo de mudanças legislativas',
        'Auditoria automática de documentos',
        'Alertas de conformidade em tempo real',
        'Relatórios personalizados para diferentes stakeholders',
      ],
      stats: [
        { label: 'Conformidade', value: '99.9%' },
        { label: 'Alertas detectados', value: '500+' },
        { label: 'Empresas protegidas', value: '200+' },
      ],
    },
    analytics: {
      image: '/images/LegalTechnology.jpg',
      capabilities: [
        'Análise preditiva de resultados judiciais',
        'Otimização de estratégias de litígio',
        'Identificação de padrões em grandes volumes de dados',
        'Dashboard interativo com insights acionáveis',
      ],
      stats: [
        { label: 'Precisão preditiva', value: '92%' },
        { label: 'Tempo economizado', value: '80%' },
        { label: 'Casos otimizados', value: '2K+' },
      ],
    },
  };

  const currentAgent = agentDetails[activeAgent as keyof typeof agentDetails];

  return (
    <section id="ai-agents" className="py-20 relative">
      {/* Background Pattern */}
      <div className="absolute inset-0 opacity-5">
        <div className="absolute inset-0" style={{
          backgroundImage: `url("https://mgx-backend-cdn.metadl.com/generate/images/877184/2026-01-01/dafffe65-b182-4c66-b32a-b3b591a02f91.png")`,
          backgroundSize: 'cover',
          backgroundPosition: 'center',
        }}></div>
      </div>

      <div className="container mx-auto px-6 relative z-10">
        {/* Header */}
        <div className="text-center mb-16">
          <div className="inline-flex items-center space-x-2 bg-gradient-to-r from-tjuridico-cyan/10 to-tjuridico-purple/10 border border-tjuridico-cyan/20 rounded-full px-4 py-2 mb-6">
            <Zap className="w-4 h-4 text-tjuridico-cyan" />
            <span className="text-tjuridico-cyan font-medium">Tecnologia de Ponta</span>
          </div>
          <h2 className="text-4xl md:text-5xl font-bold mb-6">
            <span className="text-white">Nossos </span>
            <span className="gradient-text">Agentes de IA</span>
          </h2>
          <p className="text-xl text-tjuridico-gray max-w-3xl mx-auto">
            Especialistas virtuais treinados especificamente para o direito brasileiro, 
            disponíveis 24/7 para otimizar seus processos.
          </p>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-12">
          {/* Agents List */}
          <div className="lg:col-span-1">
            <div className="space-y-4">
              {agents.map((agent) => (
                <Card
                  key={agent.id}
                  className={`glass border-white/10 cursor-pointer transition-all duration-300 hover:border-tjuridico-cyan/30 group ${
                    activeAgent === agent.id ? 'border-tjuridico-cyan/50 bg-tjuridico-cyan/5' : ''
                  }`}
                  onClick={() => setActiveAgent(agent.id)}
                  style={{ animationDelay: agent.delay }}
                >
                  <CardContent className="p-6">
                    <div className="flex items-center space-x-4">
                      <div className={`w-12 h-12 rounded-xl bg-gradient-to-br ${agent.color} flex items-center justify-center`}>
                        <agent.icon className="w-6 h-6 text-white" />
                      </div>
                      <div className="flex-1">
                        <h3 className="text-lg font-bold text-white">{agent.name}</h3>
                        <p className="text-sm text-tjuridico-gray">{agent.description}</p>
                      </div>
                      <ChevronRight className={`w-5 h-5 text-tjuridico-gray transition-transform group-hover:translate-x-1 ${
                        activeAgent === agent.id ? 'text-tjuridico-cyan' : ''
                      }`} />
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>

            {/* Stats Summary */}
            <div className="mt-8 glass rounded-2xl p-6">
              <div className="flex items-center space-x-3 mb-4">
                <Users className="w-6 h-6 text-tjuridico-cyan" />
                <h3 className="text-lg font-bold text-white">Impacto dos Agentes</h3>
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div className="text-center">
                  <div className="text-2xl font-bold text-white">60%</div>
                  <div className="text-sm text-tjuridico-gray">Mais eficiência</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-white">40%</div>
                  <div className="text-sm text-tjuridico-gray">Redução de custos</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-white">24/7</div>
                  <div className="text-sm text-tjuridico-gray">Disponibilidade</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-white">98%</div>
                  <div className="text-sm text-tjuridico-gray">Satisfação</div>
                </div>
              </div>
            </div>
          </div>

          {/* Agent Details */}
          <div className="lg:col-span-2">
            <Card className="glass border-white/10 overflow-hidden">
              <CardContent className="p-0">
                {/* Agent Header */}
                <div className="p-8 border-b border-white/10">
                  <div className="flex items-center justify-between">
                    <div>
                      <h3 className="text-2xl font-bold text-white">
                        {agents.find(a => a.id === activeAgent)?.name}
                      </h3>
                      <p className="text-tjuridico-gray">
                        {agents.find(a => a.id === activeAgent)?.description}
                      </p>
                    </div>
                    <Button className="btn-futuristic">
                      Testar Agente
                      <MessageSquare className="w-4 h-4 ml-2" />
                    </Button>
                  </div>
                </div>

                {/* Agent Content */}
                <div className="p-8">
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                    {/* Capabilities */}
                    <div>
                      <h4 className="text-lg font-bold text-white mb-4">Capacidades</h4>
                      <ul className="space-y-3">
                        {currentAgent.capabilities.map((capability, index) => (
                          <li key={index} className="flex items-start">
                            <div className="w-2 h-2 rounded-full bg-tjuridico-cyan mt-2 mr-3"></div>
                            <span className="text-tjuridico-gray">{capability}</span>
                          </li>
                        ))}
                      </ul>
                    </div>

                    {/* Stats */}
                    <div>
                      <h4 className="text-lg font-bold text-white mb-4">Estatísticas</h4>
                      <div className="space-y-4">
                        {currentAgent.stats.map((stat, index) => (
                          <div key={stat.label} className="glass rounded-xl p-4">
                            <div className="flex justify-between items-center">
                              <span className="text-tjuridico-gray">{stat.label}</span>
                              <span className="text-2xl font-bold text-white">{stat.value}</span>
                            </div>
                            <div className="mt-2 h-2 bg-white/10 rounded-full overflow-hidden">
                              <div 
                                className="h-full bg-gradient-to-r from-tjuridico-cyan to-tjuridico-purple rounded-full"
                                style={{ width: `${parseInt(stat.value) + 2}%` }}
                              ></div>
                            </div>
                          </div>
                        ))}
                      </div>
                    </div>
                  </div>

                  {/* Demo Preview */}
                  <div className="mt-8 pt-8 border-t border-white/10">
                    <h4 className="text-lg font-bold text-white mb-4">Demonstração Interativa</h4>
                    <div className="glass rounded-xl p-6">
                      <div className="flex items-center space-x-4 mb-4">
                        <div className="w-10 h-10 rounded-full bg-gradient-to-br from-tjuridico-cyan to-tjuridico-purple flex items-center justify-center">
                          <Brain className="w-5 h-5 text-white" />
                        </div>
                        <div className="flex-1">
                          <div className="text-white font-medium">Agente {agents.find(a => a.id === activeAgent)?.name}</div>
                          <div className="text-sm text-tjuridico-gray">Online • Pronto para ajudar</div>
                        </div>
                      </div>
                      <div className="space-y-3">
                        <div className="text-right">
                          <div className="inline-block bg-gradient-to-r from-tjuridico-cyan to-tjuridico-purple text-white rounded-2xl rounded-tr-none px-4 py-2 max-w-xs">
                            Como posso analisar um contrato de prestação de serviços?
                          </div>
                        </div>
                        <div>
                          <div className="inline-block bg-white/10 text-white rounded-2xl rounded-tl-none px-4 py-2 max-w-xs">
                            Posso analisar seu contrato em segundos. Basta enviar o documento ou descrever os principais pontos.
                          </div>
                        </div>
                      </div>
                      <div className="mt-4 flex space-x-2">
                        <input
                          type="text"
                          placeholder="Digite sua pergunta..."
                          className="flex-1 bg-white/5 border border-white/10 rounded-lg px-4 py-2 text-white placeholder:text-tjuridico-gray focus:outline-none focus:border-tjuridico-cyan"
                        />
                        <Button className="bg-gradient-to-r from-tjuridico-cyan to-tjuridico-purple">
                          Enviar
                        </Button>
                      </div>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </section>
  );
};

export default AIAgents;