import { useState, useEffect } from 'react';
import { Menu, X, MessageSquare, ChevronRight } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Link } from 'react-router-dom';

const Navbar = () => {
  const [isOpen, setIsOpen] = useState(false);
  const [scrolled, setScrolled] = useState(false);

  useEffect(() => {
    const handleScroll = () => {
      setScrolled(window.scrollY > 20);
    };
    window.addEventListener('scroll', handleScroll);
    return () => window.removeEventListener('scroll', handleScroll);
  }, []);

  const navItems = [
    { label: 'Início', href: '/' },
    { label: 'Serviços', href: '#services' },
    { label: 'Agentes IA', href: '#ai-agents' },
    { label: 'Planos', href: '#plans' },
    { label: 'Sobre', href: '#about' },
    { label: 'Contato', href: '#contact' },
  ];

  return (
    <nav
      className={`fixed top-0 left-0 right-0 z-50 transition-all duration-300 ${
        scrolled
          ? 'glass backdrop-blur-xl py-4 shadow-[0_10px_30px_rgba(0,0,0,0.3)]'
          : 'bg-transparent py-6'
      }`}
    >
      <div className="container mx-auto px-6">
        <div className="flex items-center justify-between">
          {/* Logo */}
          <Link to="/" className="flex items-center space-x-3 group">
            <div className="relative">
              <div className="w-10 h-10 bg-gradient-to-br from-tjuridico-cyan to-tjuridico-purple rounded-xl flex items-center justify-center">
                <span className="text-white font-bold text-lg">TJ</span>
              </div>
              <div className="absolute -inset-1 bg-gradient-to-br from-tjuridico-cyan to-tjuridico-purple rounded-xl blur opacity-30 group-hover:opacity-50 transition-opacity"></div>
            </div>
            <div className="flex flex-col">
              <span className="text-xl font-bold bg-gradient-to-r from-tjuridico-cyan to-tjuridico-purple bg-clip-text text-transparent">
                Tjuridico
              </span>
              <span className="text-xs text-tjuridico-gray">.com.br</span>
            </div>
          </Link>

          {/* Desktop Navigation */}
          <div className="hidden md:flex items-center space-x-8">
            {navItems.map((item) => (
              <a
                key={item.label}
                href={item.href}
                className="text-tjuridico-gray hover:text-white transition-colors duration-300 relative group"
              >
                {item.label}
                <span className="absolute -bottom-1 left-0 w-0 h-0.5 bg-gradient-to-r from-tjuridico-cyan to-tjuridico-purple transition-all duration-300 group-hover:w-full"></span>
              </a>
            ))}
          </div>

          {/* CTA Buttons */}
          <div className="hidden md:flex items-center space-x-4">
            <Button
              variant="outline"
              className="border-tjuridico-cyan/30 text-tjuridico-cyan hover:bg-tjuridico-cyan/10 hover:border-tjuridico-cyan transition-all duration-300"
            >
              <MessageSquare className="w-4 h-4 mr-2" />
              Chat IA
            </Button>
            <Button className="btn-futuristic">
              Começar Agora
              <ChevronRight className="w-4 h-4 ml-2" />
            </Button>
          </div>

          {/* Mobile Menu Button */}
          <button
            onClick={() => setIsOpen(!isOpen)}
            className="md:hidden text-tjuridico-gray hover:text-white transition-colors"
          >
            {isOpen ? <X className="w-6 h-6" /> : <Menu className="w-6 h-6" />}
          </button>
        </div>

        {/* Mobile Menu */}
        {isOpen && (
          <div className="md:hidden mt-6 glass rounded-2xl p-6 animate-in fade-in slide-in-from-top-4">
            <div className="flex flex-col space-y-4">
              {navItems.map((item) => (
                <a
                  key={item.label}
                  href={item.href}
                  className="text-tjuridico-gray hover:text-white py-2 px-4 rounded-lg hover:bg-white/5 transition-all duration-300"
                  onClick={() => setIsOpen(false)}
                >
                  {item.label}
                </a>
              ))}
              <div className="pt-4 border-t border-white/10 space-y-3">
                <Button
                  variant="outline"
                  className="w-full border-tjuridico-cyan/30 text-tjuridico-cyan hover:bg-tjuridico-cyan/10"
                >
                  <MessageSquare className="w-4 h-4 mr-2" />
                  Chat IA
                </Button>
                <Button className="w-full btn-futuristic">
                  Começar Agora
                  <ChevronRight className="w-4 h-4 ml-2" />
                </Button>
              </div>
            </div>
          </div>
        )}
      </div>
    </nav>
  );
};

export default Navbar;