import { useState, useRef, useEffect } from 'react';
import { MessageSquare, X, Send, Bot, User, Sparkles } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';

interface Message {
  id: number;
  text: string;
  sender: 'user' | 'bot';
  timestamp: Date;
}

const Chatbot = () => {
  const [isOpen, setIsOpen] = useState(false);
  const [messages, setMessages] = useState<Message[]>([
    {
      id: 1,
      text: 'Olá! Sou o assistente virtual da Tjuridico. Como posso ajudar você hoje?',
      sender: 'bot',
      timestamp: new Date(),
    },
  ]);
  const [inputValue, setInputValue] = useState('');
  const [isTyping, setIsTyping] = useState(false);
  const messagesEndRef = useRef<HTMLDivElement>(null);

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  };

  useEffect(() => {
    scrollToBottom();
  }, [messages]);

  const handleSendMessage = () => {
    if (!inputValue.trim()) return;

    // Add user message
    const userMessage: Message = {
      id: messages.length + 1,
      text: inputValue,
      sender: 'user',
      timestamp: new Date(),
    };

    setMessages([...messages, userMessage]);
    setInputValue('');
    setIsTyping(true);

    // Simulate bot response after delay
    setTimeout(() => {
      const responses = [
        'Entendi sua pergunta sobre análise de contratos. Posso ajudar analisando documentos específicos ou explicando como nossa IA funciona para esse caso.',
        'Para consultas jurídicas, nossa IA é treinada com a legislação brasileira atualizada. Posso pesquisar informações específicas ou conectar você com um especialista.',
        'Sobre conformidade com LGPD, temos agentes especializados que podem realizar auditorias automáticas e gerar relatórios de compliance.',
        'Para demonstrar como funciona, posso analisar um exemplo de contrato ou mostrar casos similares que já resolvemos.',
        'Posso ajudar com: análise de documentos, consultas jurídicas, due diligence, compliance LGPD, ou agendar uma demonstração personalizada.',
      ];

      const botMessage: Message = {
        id: messages.length + 2,
        text: responses[Math.floor(Math.random() * responses.length)],
        sender: 'bot',
        timestamp: new Date(),
      };

      setMessages(prev => [...prev, botMessage]);
      setIsTyping(false);
    }, 1000);
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      handleSendMessage();
    }
  };

  const quickReplies = [
    'Como funciona a análise de contratos?',
    'Quais áreas do direito cobrem?',
    'Preciso de ajuda com LGPD',
    'Agendar demonstração',
  ];

  return (
    <>
      {/* Chat Toggle Button */}
      <Button
        onClick={() => setIsOpen(!isOpen)}
        className="fixed bottom-6 right-6 w-14 h-14 rounded-full bg-gradient-to-br from-tjuridico-cyan to-tjuridico-purple shadow-lg hover:shadow-xl hover:scale-105 transition-all duration-300 z-50"
        size="icon"
      >
        {isOpen ? (
          <X className="w-6 h-6" />
        ) : (
          <MessageSquare className="w-6 h-6" />
        )}
      </Button>

      {/* Chat Window */}
      {isOpen && (
        <div className="fixed bottom-24 right-6 w-96 h-[600px] glass rounded-2xl border border-white/20 shadow-2xl overflow-hidden z-50 animate-in slide-in-from-bottom-4">
          {/* Header */}
          <div className="bg-gradient-to-r from-tjuridico-cyan/20 to-tjuridico-purple/20 border-b border-white/10 p-4">
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-3">
                <div className="w-10 h-10 rounded-full bg-gradient-to-br from-tjuridico-cyan to-tjuridico-purple flex items-center justify-center">
                  <Bot className="w-6 h-6 text-white" />
                </div>
                <div>
                  <h3 className="font-bold text-white">Assistente Tjuridico</h3>
                  <div className="flex items-center space-x-1">
                    <div className="w-2 h-2 rounded-full bg-green-400"></div>
                    <span className="text-xs text-tjuridico-gray">Online</span>
                  </div>
                </div>
              </div>
              <Button
                variant="ghost"
                size="icon"
                onClick={() => setIsOpen(false)}
                className="text-tjuridico-gray hover:text-white"
              >
                <X className="w-5 h-5" />
              </Button>
            </div>
          </div>

          {/* Messages Container */}
          <div className="flex-1 h-[400px] overflow-y-auto p-4 space-y-4">
            {messages.map((message) => (
              <div
                key={message.id}
                className={`flex ${message.sender === 'user' ? 'justify-end' : 'justify-start'}`}
              >
                <div
                  className={`max-w-[80%] rounded-2xl p-4 ${
                    message.sender === 'user'
                      ? 'bg-gradient-to-r from-tjuridico-cyan to-tjuridico-purple text-white rounded-br-none'
                      : 'glass border border-white/10 text-white rounded-bl-none'
                  }`}
                >
                  <div className="flex items-center space-x-2 mb-2">
                    {message.sender === 'bot' ? (
                      <Bot className="w-4 h-4" />
                    ) : (
                      <User className="w-4 h-4" />
                    )}
                    <span className="text-xs opacity-70">
                      {message.timestamp.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                    </span>
                  </div>
                  <p className="text-sm">{message.text}</p>
                </div>
              </div>
            ))}

            {isTyping && (
              <div className="flex justify-start">
                <div className="glass border border-white/10 rounded-2xl rounded-bl-none p-4">
                  <div className="flex items-center space-x-2">
                    <div className="flex space-x-1">
                      <div className="w-2 h-2 rounded-full bg-tjuridico-cyan animate-bounce"></div>
                      <div className="w-2 h-2 rounded-full bg-tjuridico-cyan animate-bounce animation-delay-100"></div>
                      <div className="w-2 h-2 rounded-full bg-tjuridico-cyan animate-bounce animation-delay-200"></div>
                    </div>
                    <span className="text-sm text-tjuridico-gray">Digitando...</span>
                  </div>
                </div>
              </div>
            )}

            <div ref={messagesEndRef} />
          </div>

          {/* Quick Replies */}
          <div className="p-4 border-t border-white/10">
            <div className="flex flex-wrap gap-2 mb-4">
              {quickReplies.map((reply, index) => (
                <Button
                  key={index}
                  variant="outline"
                  size="sm"
                  onClick={() => {
                    setInputValue(reply);
                    setTimeout(() => {
                      const input = document.querySelector('input[type="text"]') as HTMLInputElement;
                      input?.focus();
                    }, 100);
                  }}
                  className="text-xs border-tjuridico-cyan/30 text-tjuridico-cyan hover:bg-tjuridico-cyan/10"
                >
                  {reply}
                </Button>
              ))}
            </div>

            {/* Input Area */}
            <div className="flex space-x-2">
              <Input
                type="text"
                value={inputValue}
                onChange={(e) => setInputValue(e.target.value)}
                onKeyPress={handleKeyPress}
                placeholder="Digite sua mensagem..."
                className="flex-1 bg-white/5 border-white/10 text-white placeholder:text-tjuridico-gray focus:border-tjuridico-cyan"
              />
              <Button
                onClick={handleSendMessage}
                disabled={!inputValue.trim()}
                className="bg-gradient-to-r from-tjuridico-cyan to-tjuridico-purple hover:opacity-90"
              >
                <Send className="w-4 h-4" />
              </Button>
            </div>
          </div>

          {/* Footer */}
          <div className="p-3 border-t border-white/10 bg-black/20">
            <div className="flex items-center justify-between text-xs text-tjuridico-gray">
              <div className="flex items-center space-x-1">
                <Sparkles className="w-3 h-3" />
                <span>IA especializada em direito brasileiro</span>
              </div>
              <span>Resposta em segundos</span>
            </div>
          </div>
        </div>
      )}
    </>
  );
};

export default Chatbot;